<?php

namespace Modules\Subscription\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use Modules\Subscription\Repositories\Subscription\SubscriptionRepository;

class SubscriptionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if(module_enable_check('subscription')) {
            $options = [];
            $filterType = '';
            if ($request->has('filter')) {
                $filterType = $request->filter ?? '';
            }
            switch ($filterType) {
                case 'trash':
                    $options['onlyTrashed'] = [];
                    break;
                case 'all':
                    $options['withTrashed'] = [];
                    break;
            }
            $response = SubscriptionRepository::paginate(10, options: $options, relations: [
                'translations' => function ($query) {
                    $query->where('locale', app()->getLocale());
                }
            ]);
            $subscriptions = $response['data'] ?? [];
            $countResponse = SubscriptionRepository::trashCount();

            $countData = [
                'total' => 0,
                'published' => 0,
                'trashed' => 0
            ];

            if ($countResponse['status'] === 'success') {
                $countData = $countResponse['data']->toArray() ?? $countData;
            }
            return view('subscription::admin.subscription.index', compact('subscriptions', 'countData'));
        } else {
            return redirect()->route('lms.module', ['name' => 'subscription']);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('subscription::admin.subscription.form');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = SubscriptionRepository::save($request);
        if ($response['status'] !== 'success') {
            return response()->json($response);
        }
        return $this->jsonSuccess('Subscription has been saved successfully!', route('subscription.index'));
    }

    /**
     * Show the specified resource.
     */
    public function show($id)
    {

        $response = SubscriptionRepository::first($id,  withTrashed: true);
        $subscription =   $response['data'] ?? null;
        return view('subscription::admin.subscription.view', compact('subscription'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id, Request $request)
    {
        $locale = $request->locale ?? app()->getLocale();
        $response = SubscriptionRepository::first($id, relations: [
            'translations' => function ($query) use ($locale) {
                $query->where('locale', $locale);
            }
        ]);
        $subscription = $response['data'] ?? null;
        return view('subscription::admin.subscription.form', compact('subscription'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // Check if user has permission to edit blog categories
        if (!has_permissions($request->user(), ['edit.subscription'])) {
            return json_error('You have no permission.');
        }
        $category = SubscriptionRepository::update($id, $request);
        // Return response based on update status
        return $category['status'] !== 'success'
            ? response()->json($category)
            : $this->jsonSuccess('Subscription has been updated successfully!', route('subscription.index'));
    }
    /**
     * Change the status of the specified blog.
     */
    public function statusChange($id, Request $request): JsonResponse
    {
        // Check if user has permission to change blog status
        if (!has_permissions($request->user(), ['status.subscription'])) {
            return json_error('You have no permission.');
        }
        $response = SubscriptionRepository::statusChange($id);
        $response['url'] = route('subscription.index');
        return response()->json($response);
    }

    /**
     * Remove the specified blog Subscription from storage.
     */
    public function destroy($id, Request $request): JsonResponse
    {
        // Check if user has permission to delete subscriptions
        if (!has_permissions($request->user(), ['delete.subscription'])) {
            return json_error('You have no permission.');
        }

        $category = SubscriptionRepository::delete($id);
        $category['url'] = route('subscription.index');
        return response()->json($category);
    }

    public function restore(int $id, Request $request)
    {
        // Check user permission to delete a category
        if (!has_permissions($request->user(), ['delete.subscription'])) {
            return json_error('You have no permission.');
        }
        $response = SubscriptionRepository::restore(id: $id);
        $response['url'] = route('subscription.index');
        return response()->json($response);
    }
}
